// src/shader.rs

use web_sys::{WebGlProgram, WebGlShader};
use crate::gl::GLContext;
use web_sys::WebGlRenderingContext as GL;

const VERTEX_SRC: &str = include_str!("../shaders/vertex.glsl");

const FRAG_SRC: &str = include_str!("../shaders/fragment.glsl");

#[derive(Clone)]
pub struct ShaderProgram {
    pub program: WebGlProgram,
    pub position: u32,
    pub color: u32,
    pub scale_mat3:       web_sys::WebGlUniformLocation,
    pub translate_mat3:   web_sys::WebGlUniformLocation,
    pub perspective_mat3: web_sys::WebGlUniformLocation,
}

impl ShaderProgram {
    pub fn new(gl: &GLContext) -> Self {
        let vert    = compile_shader(&gl.gl, GL::VERTEX_SHADER, VERTEX_SRC);
        let frag    = compile_shader(&gl.gl, GL::FRAGMENT_SHADER, FRAG_SRC);
        let program = link_program(&gl.gl, &vert, &frag);

        let position = gl.gl.get_attrib_location(&program, "position") as u32;
        let color    = gl.gl.get_attrib_location(&program, "color")    as u32;

        let scale_mat3 = gl.gl
            .get_uniform_location(&program, "scale_mat3")
            .expect("Uniform 'scale_mat3' not found");

        let translate_mat3 = gl.gl
            .get_uniform_location(&program, "translate_mat3")
            .expect("Uniform 'translate_mat3' not found");

        let perspective_mat3 = gl.gl
            .get_uniform_location(&program, "perspective_mat3")
            .expect("Uniform 'perspective_mat3' not found");

        Self {
            program,
            position,
            color,
            scale_mat3,
            translate_mat3,
            perspective_mat3,
        }
    }
}

fn compile_shader(gl: &GL, kind: u32, src: &str) -> WebGlShader {
    let shader = gl.create_shader(kind).unwrap();
    gl.shader_source(&shader, src);
    gl.compile_shader(&shader);

    if gl.get_shader_parameter(&shader, GL::COMPILE_STATUS).as_bool().unwrap() {
        shader
    } else {
        panic!("Shader compilation failed: {}", gl.get_shader_info_log(&shader).unwrap());
    }
}

fn link_program(gl: &GL, vert: &WebGlShader, frag: &WebGlShader) -> WebGlProgram {
    let program = gl.create_program().unwrap();
    gl.attach_shader(&program, vert);
    gl.attach_shader(&program, frag);
    gl.link_program(&program);

    if gl.get_program_parameter(&program, GL::LINK_STATUS).as_bool().unwrap() {
        program
    } else {
        panic!("Shader linking failed: {}", gl.get_program_info_log(&program).unwrap());
    }
}
