// src/lib.rs

mod gl;            // WebGL context + helper
mod shader;        // Shader compile/link
mod geometry;      // Vertex data + matrix math

use wasm_bindgen::prelude::*;
use std::cell::RefCell;
use std::rc::Rc;
use web_sys::WebGlRenderingContext as GL;

#[wasm_bindgen(start)]
pub fn start() {
    run();
}

pub fn run() {
    let gl = gl::GLContext::from_canvas_id("canvas");
    let program = shader::ShaderProgram::new(&gl);
    let start_time = js_sys::Date::now();

    let vertices = geometry::colored_square_vertices();
    let buffer = gl.create_buffer_with_data(&vertices);

    gl.use_program(&program);
    gl.bind_buffer(&buffer);
    gl.configure_attribute(&program.position, 3, 6, 0);
    gl.configure_attribute(&program.color, 3, 6, 3);

    let anim = Rc::new(RefCell::new(None::<Closure<dyn FnMut()>>));
    let anim_clone = anim.clone();

    let window = web_sys::window().unwrap();
    let window_clone = window.clone();
    let screen_width  = window.inner_width() .unwrap().as_f64().unwrap() as f32;
    let screen_height = window.inner_height().unwrap().as_f64().unwrap() as f32;

    *anim_clone.borrow_mut() = Some(Closure::new({
        let gl      = gl        .clone();
        let program = program   .clone();
        let anim    = anim_clone.clone();
        let start_time = start_time;

        move || {
            gl.clear_color(0.0, 0.0, 0.0, 1.0);
            gl.clear(GL::COLOR_BUFFER_BIT);

            let now = js_sys::Date::now();
            let time = ((now - start_time) / 1000.0) as f32;

            let scale_mat3       = geometry::scale_mat3(1.0, 1.0);
            let translate_mat3   = geometry::translate_mat3(0.0, 0.0);
            let perspective_mat3 = geometry::one_point_perspective_mat3(0.0, 0.0, 0.0);
            

            gl.set_uniform_mat3(&program.scale_mat3,       &scale_mat3);
            gl.set_uniform_mat3(&program.translate_mat3,   &translate_mat3);
            gl.set_uniform_mat3(&program.perspective_mat3, &perspective_mat3);
            gl.draw_triangles(6);

            window_clone
                .request_animation_frame(
                    anim.borrow().as_ref().unwrap().as_ref().unchecked_ref(),
                )
                .unwrap();
        }
    }));

    window
        .request_animation_frame(
            anim.borrow().as_ref().unwrap().as_ref().unchecked_ref(),
        )
        .unwrap();
} 
