// src/gl.rs

use web_sys::{
    WebGlRenderingContext as GL, 
    WebGlBuffer, 
    HtmlCanvasElement
};
use wasm_bindgen::JsCast;

#[derive(Clone)]
pub struct GLContext {
    pub gl: GL,
}

impl GLContext {
    pub fn from_canvas_id(id: &str) -> Self {
        let doc = web_sys::window()
            .unwrap()
            .document()
            .unwrap();
        let canvas: HtmlCanvasElement = doc
            .get_element_by_id(id)
            .unwrap()
            .dyn_into()
            .unwrap();

        let gl: GL = canvas
            .get_context("webgl")
            .unwrap()
            .unwrap()
            .dyn_into::<GL>()
            .unwrap();

        let window = web_sys::window().unwrap();
        let device_pixel_ratio = window.device_pixel_ratio();
        let width  = (window.inner_width() .unwrap().as_f64().unwrap() * device_pixel_ratio) as u32;
        let height = (window.inner_height().unwrap().as_f64().unwrap() * device_pixel_ratio) as u32;

        canvas.set_width(width);
        canvas.set_height(height);

        gl.viewport(0, 0, width as i32, height as i32);

        Self { gl }
    }

    pub fn create_buffer_with_data(&self, data: &[f32]) -> WebGlBuffer {
        let buffer = self.gl.create_buffer().unwrap();
        self.gl.bind_buffer(GL::ARRAY_BUFFER, Some(&buffer));

        unsafe {
            let array = js_sys::Float32Array::view(data);
            self.gl.buffer_data_with_array_buffer_view(
                GL::ARRAY_BUFFER,
                &array,
                GL::STATIC_DRAW,
            );
        }
        buffer
    }

    pub fn bind_buffer(&self, buffer: &WebGlBuffer) {
        self.gl.bind_buffer(GL::ARRAY_BUFFER, Some(buffer));
    }

    pub fn configure_attribute(&self, attrib: &u32, size: i32, stride: i32, offset: i32) {
        self.gl.enable_vertex_attrib_array(*attrib);
        self.gl.vertex_attrib_pointer_with_i32(
            *attrib,
            size,
            GL::FLOAT,
            false,
            stride * std::mem::size_of::<f32>() as i32,
            offset * std::mem::size_of::<f32>() as i32,
        );
    }

    pub fn use_program(&self, program: &crate::shader::ShaderProgram) {
        self.gl.use_program(Some(&program.program));
    }

    pub fn clear_color(&self, r: f32, g: f32, b: f32, a: f32) {
        self.gl.clear_color(r, g, b, a);
    }

    pub fn clear(&self, mask: u32) {
        self.gl.clear(mask);
    }

    pub fn draw_triangles(&self, count: i32) {
        self.gl.draw_arrays(GL::TRIANGLES, 0, count);
    }

    pub fn set_uniform_mat3(&self, uniform: &web_sys::WebGlUniformLocation, data: &[f32; 9]) {
        self.gl.uniform_matrix3fv_with_f32_array(Some(uniform), false, data);
    }
}
